#!/usr/bin/env node

/*
    CLI implementation for Rom Patcher JS
    https://github.com/marcrobledo/RomPatcher.js
    by Marc Robledo, released under MIT license: https://github.com/marcrobledo/RomPatcher.js/blob/master/LICENSE

    Usage:
        Install needed dependencies:
        > npm install

        Patch a ROM:
        > node index.js patch "my_rom.bin" "my_patch.ips" -o "patched_rom.bin"

        Create a patch from two ROMs:
        > node index.js create "original_rom.bin" "modified_rom.bin"

        For more options:
        > node index.js patch --help
        > node index.js create --help
*/

const fs = require('fs');
const path = require('path');
const chalk = require('chalk');
const { program } = require('commander');
const RomPatcher = require('./RomPatcher');

program
    .command('patch')
    .description('patches a ROM')
    .argument('<rom_file>','the ROM file that will be patched')
    .argument('<patch_file>', 'the patch to apply')
    .option('-v, --validate-checksum','should validate checksum')
    .option('-h1, --add-header','adds a temporary header to the provided ROM for patches that require headered ROMs')
    .option('-h0, --remove-header','removes ROM header temporarily for patches that require headerless ROMs')
    .option('-f, --fix-checksum','fixes any known ROM header checksum if possible')
    .option('-s, --output-suffix','add a (patched) suffix to output ROM file name')
    .option('-o, --output <destination>', 'destination path for the patched ROM file')
    .action(function(romPath, patchPath, options) {
        try{
            const romFile = new BinFile(romPath);
            const patchFile = new BinFile(patchPath);

            const patch = RomPatcher.parsePatchFile(patchFile);
            if(!patch)
                throw new Error('Invalid patch file');

            const patchedRom = RomPatcher.applyPatch(romFile, patch, options);

            // Determine output filename if user provided --output
            if (options.output) {
                let outStat = null;
                try { outStat = fs.statSync(options.output); } catch (e) { outStat = null; }

                if (outStat && outStat.isDirectory()) {
                    const baseName = path.basename(patchedRom.fileName || romFile.getName());
                    patchedRom.fileName = path.join(options.output, baseName);
                } else {
                    // Treat as full file path (may overwrite)
                    patchedRom.fileName = options.output;
                }
            } else if (options.outputSuffix) {
                // Add "(patched)" before extension if requested
                const originalName = patchedRom.fileName || romFile.getName();
                const extIndex = originalName.lastIndexOf('.');
                if (extIndex !== -1) {
                    patchedRom.fileName = originalName.slice(0, extIndex) + '(patched)' + originalName.slice(extIndex);
                } else {
                    patchedRom.fileName = originalName + '(patched)';
                }
            }

            patchedRom.save();
            console.log(chalk.green('successfully saved to ' + patchedRom.fileName));
        }catch(err){
            console.log(chalk.bgRed('error: ' + err.message));
        }
    });

program
    .command('create')
    .description('creates a patch based on two ROMs')
    .argument('<original_rom_file>', 'the original ROM')
    .argument('<modified_rom_file>','the modified ROM')
    .option('-f, --format <format>','patch format (allowed values: ips [default], bps, ppf, ups, aps, rup, ebp)')
    .action(function(originalRomPath, modifiedRomPath, options) {
        try{
            const originalFile = new BinFile(originalRomPath);
            const modifiedFile = new BinFile(modifiedRomPath);

            const patch = RomPatcher.createPatch(originalFile, modifiedFile, options.format);
            const patchFile = patch.export();
            patchFile.setName(modifiedFile.getName());
            patchFile.save();
            console.log(chalk.green('patch created: ' + (patchFile.fileName || patchFile.getName())));
        }catch(err){
            console.log(chalk.bgBlue('Error: ' + err.message));
        }
    });

program.parse();
